/*
 * Decompiled with CFR 0.152.
 */
package org.bouncycastle.crypto.util;

import java.io.IOException;
import java.math.BigInteger;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.nist.NISTNamedCurves;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.pkcs.RSAPrivateKey;
import org.bouncycastle.asn1.sec.ECPrivateKey;
import org.bouncycastle.asn1.x9.X962Parameters;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.crypto.CryptoServicesRegistrar;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.params.DSAParameters;
import org.bouncycastle.crypto.params.DSAPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECNamedDomainParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.Ed25519PrivateKeyParameters;
import org.bouncycastle.crypto.params.Ed25519PublicKeyParameters;
import org.bouncycastle.crypto.params.RSAPrivateCrtKeyParameters;
import org.bouncycastle.crypto.util.OpenSSHPublicKeyUtil;
import org.bouncycastle.crypto.util.PrivateKeyInfoFactory;
import org.bouncycastle.crypto.util.SSHBuffer;
import org.bouncycastle.crypto.util.SSHBuilder;
import org.bouncycastle.crypto.util.SSHNamedCurves;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.BigIntegers;
import org.bouncycastle.util.Strings;

public class OpenSSHPrivateKeyUtil {
    static final byte[] AUTH_MAGIC = Strings.toByteArray("openssh-key-v1\u0000");

    private OpenSSHPrivateKeyUtil() {
    }

    public static byte[] encodePrivateKey(AsymmetricKeyParameter params) throws IOException {
        if (params == null) {
            throw new IllegalArgumentException("param is null");
        }
        if (params instanceof RSAPrivateCrtKeyParameters) {
            PrivateKeyInfo pInfo = PrivateKeyInfoFactory.createPrivateKeyInfo(params);
            return pInfo.parsePrivateKey().toASN1Primitive().getEncoded();
        }
        if (params instanceof ECPrivateKeyParameters) {
            PrivateKeyInfo pInfo = PrivateKeyInfoFactory.createPrivateKeyInfo(params);
            return pInfo.parsePrivateKey().toASN1Primitive().getEncoded();
        }
        if (params instanceof DSAPrivateKeyParameters) {
            DSAPrivateKeyParameters dsaPrivKey = (DSAPrivateKeyParameters)params;
            DSAParameters dsaParams = dsaPrivKey.getParameters();
            ASN1EncodableVector vec = new ASN1EncodableVector();
            vec.add(new ASN1Integer(0L));
            vec.add(new ASN1Integer(dsaParams.getP()));
            vec.add(new ASN1Integer(dsaParams.getQ()));
            vec.add(new ASN1Integer(dsaParams.getG()));
            BigInteger pubKey = dsaParams.getG().modPow(dsaPrivKey.getX(), dsaParams.getP());
            vec.add(new ASN1Integer(pubKey));
            vec.add(new ASN1Integer(dsaPrivKey.getX()));
            try {
                return new DERSequence(vec).getEncoded();
            }
            catch (Exception ex) {
                throw new IllegalStateException("unable to encode DSAPrivateKeyParameters " + ex.getMessage());
            }
        }
        if (params instanceof Ed25519PrivateKeyParameters) {
            Ed25519PublicKeyParameters publicKeyParameters = ((Ed25519PrivateKeyParameters)params).generatePublicKey();
            SSHBuilder builder = new SSHBuilder();
            builder.writeBytes(AUTH_MAGIC);
            builder.writeString("none");
            builder.writeString("none");
            builder.writeString("");
            builder.u32(1);
            byte[] pkEncoded = OpenSSHPublicKeyUtil.encodePublicKey(publicKeyParameters);
            builder.writeBlock(pkEncoded);
            SSHBuilder pkBuild = new SSHBuilder();
            int checkint = CryptoServicesRegistrar.getSecureRandom().nextInt();
            pkBuild.u32(checkint);
            pkBuild.u32(checkint);
            pkBuild.writeString("ssh-ed25519");
            byte[] pubKeyEncoded = publicKeyParameters.getEncoded();
            pkBuild.writeBlock(pubKeyEncoded);
            pkBuild.writeBlock(Arrays.concatenate(((Ed25519PrivateKeyParameters)params).getEncoded(), pubKeyEncoded));
            pkBuild.writeString("");
            builder.writeBlock(pkBuild.getPaddedBytes());
            return builder.getBytes();
        }
        throw new IllegalArgumentException("unable to convert " + params.getClass().getName() + " to openssh private key");
    }

    public static AsymmetricKeyParameter parsePrivateKeyBlob(byte[] blob) {
        AsymmetricKeyParameter result = null;
        if (blob[0] == 48) {
            ASN1Sequence sequence = ASN1Sequence.getInstance(blob);
            if (sequence.size() == 6) {
                if (OpenSSHPrivateKeyUtil.allIntegers(sequence) && ((ASN1Integer)sequence.getObjectAt(0)).getPositiveValue().equals(BigIntegers.ZERO)) {
                    result = new DSAPrivateKeyParameters(((ASN1Integer)sequence.getObjectAt(5)).getPositiveValue(), new DSAParameters(((ASN1Integer)sequence.getObjectAt(1)).getPositiveValue(), ((ASN1Integer)sequence.getObjectAt(2)).getPositiveValue(), ((ASN1Integer)sequence.getObjectAt(3)).getPositiveValue()));
                }
            } else if (sequence.size() == 9) {
                if (OpenSSHPrivateKeyUtil.allIntegers(sequence) && ((ASN1Integer)sequence.getObjectAt(0)).getPositiveValue().equals(BigIntegers.ZERO)) {
                    RSAPrivateKey rsaPrivateKey = RSAPrivateKey.getInstance(sequence);
                    result = new RSAPrivateCrtKeyParameters(rsaPrivateKey.getModulus(), rsaPrivateKey.getPublicExponent(), rsaPrivateKey.getPrivateExponent(), rsaPrivateKey.getPrime1(), rsaPrivateKey.getPrime2(), rsaPrivateKey.getExponent1(), rsaPrivateKey.getExponent2(), rsaPrivateKey.getCoefficient());
                }
            } else if (sequence.size() == 4 && sequence.getObjectAt(3) instanceof ASN1TaggedObject && sequence.getObjectAt(2) instanceof ASN1TaggedObject) {
                ECDomainParameters domainParams;
                ECPrivateKey ecPrivateKey = ECPrivateKey.getInstance(sequence);
                X962Parameters parameters = X962Parameters.getInstance(ecPrivateKey.getParametersObject().toASN1Primitive());
                if (parameters.isNamedCurve()) {
                    ASN1ObjectIdentifier oid = ASN1ObjectIdentifier.getInstance(parameters.getParameters());
                    domainParams = ECNamedDomainParameters.lookup(oid);
                } else {
                    X9ECParameters x9 = X9ECParameters.getInstance(parameters.getParameters());
                    domainParams = new ECDomainParameters(x9);
                }
                BigInteger d = ecPrivateKey.getKey();
                result = new ECPrivateKeyParameters(d, domainParams);
            }
        } else {
            int check2;
            SSHBuffer kIn = new SSHBuffer(AUTH_MAGIC, blob);
            String cipherName = kIn.readString();
            if (!"none".equals(cipherName)) {
                throw new IllegalStateException("encrypted keys not supported");
            }
            kIn.skipBlock();
            kIn.skipBlock();
            int publicKeyCount = kIn.readU32();
            if (publicKeyCount != 1) {
                throw new IllegalStateException("multiple keys not supported");
            }
            OpenSSHPublicKeyUtil.parsePublicKey(kIn.readBlock());
            byte[] privateKeyBlock = kIn.readPaddedBlock();
            if (kIn.hasRemaining()) {
                throw new IllegalArgumentException("decoded key has trailing data");
            }
            SSHBuffer pkIn = new SSHBuffer(privateKeyBlock);
            int check1 = pkIn.readU32();
            if (check1 != (check2 = pkIn.readU32())) {
                throw new IllegalStateException("private key check values are not the same");
            }
            String keyType = pkIn.readString();
            if ("ssh-ed25519".equals(keyType)) {
                pkIn.readBlock();
                byte[] edPrivateKey = pkIn.readBlock();
                if (edPrivateKey.length != 64) {
                    throw new IllegalStateException("private key value of wrong length");
                }
                result = new Ed25519PrivateKeyParameters(edPrivateKey, 0);
            } else if (keyType.startsWith("ecdsa")) {
                ASN1ObjectIdentifier oid = SSHNamedCurves.getByName(Strings.fromByteArray(pkIn.readBlock()));
                if (oid == null) {
                    throw new IllegalStateException("OID not found for: " + keyType);
                }
                X9ECParameters curveParams = NISTNamedCurves.getByOID(oid);
                if (curveParams == null) {
                    throw new IllegalStateException("Curve not found for: " + oid);
                }
                pkIn.readBlock();
                byte[] privKey = pkIn.readBlock();
                result = new ECPrivateKeyParameters(new BigInteger(1, privKey), (ECDomainParameters)new ECNamedDomainParameters(oid, curveParams));
            } else if (keyType.startsWith("ssh-rsa")) {
                BigInteger modulus = new BigInteger(1, pkIn.readBlock());
                BigInteger pubExp = new BigInteger(1, pkIn.readBlock());
                BigInteger privExp = new BigInteger(1, pkIn.readBlock());
                BigInteger coef = new BigInteger(1, pkIn.readBlock());
                BigInteger p = new BigInteger(1, pkIn.readBlock());
                BigInteger q = new BigInteger(1, pkIn.readBlock());
                BigInteger pSub1 = p.subtract(BigIntegers.ONE);
                BigInteger qSub1 = q.subtract(BigIntegers.ONE);
                BigInteger dP = privExp.remainder(pSub1);
                BigInteger dQ = privExp.remainder(qSub1);
                result = new RSAPrivateCrtKeyParameters(modulus, pubExp, privExp, p, q, dP, dQ, coef);
            }
            pkIn.skipBlock();
            if (pkIn.hasRemaining()) {
                throw new IllegalArgumentException("private key block has trailing data");
            }
        }
        if (result == null) {
            throw new IllegalArgumentException("unable to parse key");
        }
        return result;
    }

    private static boolean allIntegers(ASN1Sequence sequence) {
        int t = 0;
        while (t < sequence.size()) {
            if (!(sequence.getObjectAt(t) instanceof ASN1Integer)) {
                return false;
            }
            ++t;
        }
        return true;
    }
}

